/************************************************************************
  Ambitious Light 2015 (c) by sinus@sinsuweb.de
  an Arduino Due based ambient TV light
  main Features are:
  -> capture and decoding of analog RGB component video plus sync
  -> color interpolation for left, right and top RGB LED chains
  -> color transformation (brightness, gamma, saturation etc.)
  -> undersampling 38 x 38 pixels
  -> 80 ms / 12.5 fps refresh rate
  -> averaging in time domain ("afterglow" effect)
  -> intelligent combination of border colors and central colors
  -> automatic detection of 16:9 or 4:3 content
  -> TFT display for parameter menu and debugging
  -> parameter storage on SD card due to the lack of an EEPROM
  
 ************************************************************************  
  TFT, SD and NeoPixel API by Adafruit Industries:
   
  This is an example sketch for the Adafruit 1.8" SPI display.
  This library works with the Adafruit 1.8" TFT Breakout w/SD card
  ----> http://www.adafruit.com/products/358
  as well as Adafruit raw 1.8" TFT display
  ----> http://www.adafruit.com/products/618
 
  Check out the links above for our tutorials and wiring diagrams
  These displays use SPI to communicate, 4 or 5 pins are required to
  interface (RST is optional)
  Adafruit invests time and resources providing this open source code,
  please support Adafruit and open-source hardware by purchasing
  products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries.
  MIT license, all text above must be included in any redistribution
 ************************************************************************

 History:
 
  2015-06-19 
  - Analog Profiling
  
  2015-06-20
  - VSYNC
 
  2015-06-21
  - HSYNC und LineRead
  
  2015-06-22
  - ReadAnalogEx von 12 auf 8 bit herabgesetzt,
    um Arrays von Word auf Byte reduzieren zu können

  2015-07-19 
  - Hardware finished
  - Speed-Opti

  2015-07-23
  - Linear-Interpolation für LED-Kette
  - eigenes Gamma für LED-Kette
  
  2015-07-25
  - Interpolation optimiert
  - SampleTime wird nur bei Start ermittelt
  - "eps" bestimmt zeitl. Glättung

  2015-07-28 
  - Saturation, Hue, Brightness, Color for LED-Output
  - 16:9 Detection

  2015-08-01
  - alte SD-Card 500 MB benutzt für Param-Speicherung, da kein EEPROM verfügbar
  - Params generisch

  2015-08-02
  - Params für TopChainCnt, SideChainCnt, 16:9 topLine, 16:9 height ergänzt
  - echtes Averaging ergänzt
  - Poti-Werte nur übernehmen, wenn Änderung > 5% erkannt wurde

  2015-08-03 
  - PauseMode blue
  - Opti: WhiteLevel nur in UpdateTFT berechnen

  2015-08-11
  - Frame-Latency & Hue-Einstellung entfernt, nicht nötig
  - ReadSync & ReadLine optmiert (nur jede 2. Field-Zeile reicht)

  2015-08-13
  - ProcessLEDfast() implementiert
  - CenterColor begonnen

  2015-08-14
  - CenterColor-Averaging
  - CenterColor wird appliziert
  - frameRate immernoch 12.5 fps
  
  Erkenntnisse:
  - nur 96 kB dyn. Speicher vorhanden -> limitiert Array-Möglichkeiten
  - da 1.5 us SampleTime kein glattes Teilerverhältnis zu 64 us bilden,
    gibt es einen horizontalen Abtast-Fehler
  - es werden nur 304 statt 305 Zeilen pro Field übertragen (?)
  - RGB-Component-Signal enthält kein Sync -> künstlich per Comperator ergänzt
  - Sync-Channel (Composite) ist nicht DC-gekoppelt -> Klemmschaltung nötig
  - ab 3. Zeile kann Teletext enthalten sein
  - Array-Zugriffe dauern
  - LEDs benötigen absenkendes Gamma, getrennt vom TFT
  - Arduino Due hat keinen EEPROM -> Params werden auf SD-Card gespeichert
  
 ****************************************************/
 
#include "wiring_analogEx.h"
#include "RunningMedian.h"
#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library
#include <SPI.h>
#include <Adafruit_NeoPixel.h>
#include <SD.h>

//#define sclk 13
//#define mosi 11
#define cs   10
#define dc   9
#define rst  8 

// set up variables using the SD utility library functions:
//Sd2Card card;
//SdVolume volume;
//SdFile root;
File paramFile;

const int chipSelect = 12;
const float voltageGain = 2.180; // müsste eigentlich 1.0 + 7.5/2.2 = 4.4 sein, aber geringer wegen Vorwiderstand-Aufteilung und Freqenzabh.  // 1 + R2/R1 bedingt durch Verstärkung TLC074 
const word VSYNC = 0;
const word HSYNC_CAPTURE = 1;
const word HSYNC_DECODE  = 2;
const byte RM_BLANK = 0;
const byte RM_TEST_PATTERN = 1;
byte valG[14200];   // reicht für mind. 19.76 ms @ sampleTime 1.4 us
byte valR[14200];  
byte valB[14200];  
word picR[40][38];  // bei sampleTime 1.4 us entstehen max. 38 pixel horizontal, vertikal 4-Zeilen-Average: 304/4 = 76 
word picG[40][38];
word picB[40][38];
int analogPinButton = 0;
int analogPinPoti   = 1;
int analogPinGreen  = 2;
int analogPinRed    = 3;
int analogPinBlue   = 4;
unsigned long t1,t2,t3,t4 = 0;
unsigned long sampleTime_ns = 0; // real >= 1.5 us
RunningMedian<word,5> syncLevelMedianR; 
RunningMedian<word,5> syncLevelMedianG; 
RunningMedian<word,5> syncLevelMedianB; 
RunningMedian<word,5> potiMedian;
word syncLevel[3] = {0,0,0};
word blackLevel[3] = {0,0,0};
word realBlackLevel[3] = {0,0,0};
word syncThreshold[3] = {0,0,0};
bool VSYNCok[3] = {false,false,false};
bool HSYNCok[3] = {false,false,false};
unsigned long missedVSYNCcount[3] = {0,0,0};
unsigned long missedHSYNCcount[3] = {0,0,0};
word VSYNCwait_us[3] = {0,0,0};
byte gammaTableTFT[256];
byte gammaTableLED[256];
float gamma_value_TFT = 2.2; 
unsigned long backPorchSampleCount = 0;
word pixPerLine = 0;
word samplesPerLine = 0;
byte leftChain[10][40][3];   // [Avrg][LEDcnt][R/G/B]  -> Avrg: 10 Einzelwerte
byte rightChain[10][40][3];  
byte topChain[10][40][3]; 
word centerColor[10][3];
byte leftChainOut[40][3];   // Avrg-Output
byte rightChainOut[40][3];  
byte topChainOut[40][3];
byte centerColorOut[3]; 
byte centerV;
byte sideChainInterpolationPos[38];
float sideChainInterpolationDelta[38];
byte topChainInterpolationPos[40];
float topChainInterpolationDelta[40];
unsigned long frameNr = 0;
unsigned long frameCnt169 = 0;
bool is169 = false; // 16:9 ?
byte topLine = 0;
bool doUpdateTFT = false;
bool buttonPressed = false;
word potiValue = 0;
byte selectedParam = 0;
float selectedParamPotiOrig = 999;
bool SDcardOK = false;
float params[17][3];  // value/min/max  (DisplayMode=1, InputContrast, Gamma, Saturation, Brightness, Hue, Red, Green, Blue, Averaging, Latency=11)
String paramNames[17];
const String paramfileName = "params.dat";
bool triggerRepaint = true;
byte avrgIndex = 0;
unsigned long tLastRepaintEvent = 0;
word lastFPS = 0;

// Option 2: must use the hardware SPI pins (for UNO thats sclk = 13 and sid = 11) and pin 10 must be an output.
// This is much faster - also required if you want to use the microSD card (see the image drawing example)
Adafruit_ST7735 tft = Adafruit_ST7735(cs, dc, rst);
Adafruit_NeoPixel strip = Adafruit_NeoPixel(120, 11 /*PIN*/, NEO_GRB + NEO_KHZ800);  // topChainCnt + sideChainCnt *2

void setup(void) {

  // ChipSelect für SD-Card
  pinMode(12, OUTPUT);

  // If your TFT's plastic wrap has a Black Tab, use the following:
  tft.initR(INITR_BLACKTAB);   // initialize a ST7735S chip, black tab

  tft.setRotation(1);              // Landscape
  tft.fillScreen(ST7735_BLACK);    // Black
  tft.setTextSize(1);              // TextSize
  tft.setTextColor(ST7735_WHITE);  // Textcolor

  // splash
  tft.setCursor(36, 58);
  tft.setTextColor(ST7735_WHITE);
  tft.println("Ambitious Light");
  delay(1500);
  
  ///tft.setCursor(36, 68);
  ///if (card.init(SPI_HALF_SPEED, chipSelect)) { 
  ///                                             if (volume.init(card)) { SDcardOK = SD.begin(chipSelect); } 
  ///                                                               else { SDcardOK = false;                }  // keine Partition
  ///                                           }
  ///                                      else { SDcardOK = true;              }
                                     
  SDcardOK = SD.begin(chipSelect);                                        
  if (!SDcardOK) { tft.setTextColor(ST7735_RED); tft.println("SD card failed"); delay(3000); }                             
  
  // Param Defaults initialisieren, Value, Min, Max, Name
  params[0][0]  = 0;     params[0][1]  = 0;     params[0][2]  = 1.0;   paramNames[0]  = "";               // OFF, not used
  params[1][0]  = 1;     params[1][1]  = 0;     params[1][2]  = 1.0;   paramNames[1]  = "Display Mode";   // DisplayMode
  params[2][0]  = 0.5;   params[2][1]  = 0;     params[2][2]  = 2.0;   paramNames[2]  = "Input Contrast"; // Input Contrast
  params[3][0]  = 0.4;   params[3][1]  = 0.1;   params[3][2]  = 2.0;   paramNames[3]  = "Gamma";          // Gamma
  params[4][0]  = 1.0;   params[4][1]  = 0;     params[4][2]  = 1.0;   paramNames[4]  = "Saturation";     // Saturation
  params[5][0]  = 1.0;   params[5][1]  = 0;     params[5][2]  = 1.0;   paramNames[5]  = "Brightness";     // Brightness
  params[6][0]  = 1.0;   params[6][1]  = 0;     params[6][2]  = 1.0;   paramNames[6]  = "Hue";            // Hue
  params[7][0]  = 1.0;   params[7][1]  = 0;     params[7][2]  = 1.0;   paramNames[7]  = "Red";            // Red
  params[8][0]  = 1.0;   params[8][1]  = 0;     params[8][2]  = 1.0;   paramNames[8]  = "Green";          // Green   
  params[9][0]  = 1.0;   params[9][1]  = 0;     params[9][2]  = 1.0;   paramNames[9]  = "Blue";           // Blue
  params[10][0] = 3.0;   params[10][1] = 1.0;   params[10][2] = 10.0;  paramNames[10] = "Averaging";      // Averaging
  params[11][0] = 1.0;   params[11][1] = 0;     params[11][2] = 1.0;   paramNames[11] = "Center Amount";  // Center Amount
  params[12][0] = 31;    params[12][1] = 5;     params[12][2] = 40;    paramNames[12] = "TopChain Cnt";    // TopChainCnt
  params[13][0] = 16;    params[13][1] = 5;     params[13][2] = 40;    paramNames[13] = "SideChain Cnt";   // SideChainCnt
  params[14][0] = 5;     params[14][1] = 0;     params[14][2] = 8;     paramNames[14] = "16:9 TopLine";    // 16:9 TopLine
  params[15][0] = 28;    params[15][1] = 5;     params[15][2] = 32;    paramNames[15] = "16:9 Height";     // 16:9 Height
  params[16][0] = 0;     params[16][1] = 0;     params[16][2] = 1.0;   paramNames[16] = "reserved";        // reserved
    
  // ggf. Params laden
  LoadParamsFromSDCard();

  syncLevelMedianR.clear();                             // Mittelwert-Array für SyncLevel, normalerweise -0.3V
  syncLevelMedianG.clear();   
  syncLevelMedianB.clear();   
  potiMedian.clear();

  // Bildzwischenspeicher leeren
  ResetPic(RM_BLANK);
  
  //REG_ADC_MR = (REG_ADC_MR & 0xFFF0FFFF) | 0x00020000;  // ADC STARTUP time, fixed ab Arduino 1.5.5
  analogReadResolution(12);                               // ADC 12 bit
  REG_ADC_MR = (REG_ADC_MR & 0xFFFFFF0F) | 0x00000080;    // enable FREERUN mode
   
  // Timings ermitteln, SampleTime ca. 1500 ns 
  ProfileSampleTime(0, analogPinRed, valR);

  // Lookups vorberechnen
  PrepareLookUps();
  
  // LED-Kette
  strip.begin();
  strip.show(); // Initialize all pixels to 'off'
  
}

// ****** Main Loop ******

void loop() {

  // Update, alle 30 Frames oder wenn Settings-Mode aktiv
  doUpdateTFT = (((selectedParam == 0) && (frameNr%30==0)) || 
                 ((selectedParam > 0) && (triggerRepaint)));  
     
  if (!doUpdateTFT) { t3 = micros(); }

  // gelesenes Bild zurücksetzen
  ResetPic(RM_BLANK);  //RM_TEST_PATTERN); 

  // Abstand zw. Samples & SyncVoltage (ca.-0.3V) ermitteln
  ProfileSyncLevel(0, analogPinRed,   valR);  
  ProfileSyncLevel(1, analogPinGreen, valG);  
  ProfileSyncLevel(2, analogPinBlue,  valB);  

  // Bilddaten einlesen
  CaptureData(0, analogPinRed,   picR, valR);
  CaptureData(1, analogPinGreen, picG, valG);
  CaptureData(2, analogPinBlue,  picB, valB);
   
  // Zeilen dekodieren 
  DecodeData(0, analogPinRed,   picR, valR);
  DecodeData(1, analogPinGreen, picG, valG);
  DecodeData(2, analogPinBlue,  picB, valB);   

  // Bild aufbereiten, nur wenn TFT-Update ansteht komplett
  TransformPic(doUpdateTFT);
  
  // LED-Kette updaten
  UpdateLEDs();
       
  // Button & Poti
  ReadKnobs();

  if (!doUpdateTFT) { t4 = micros(); }
         
  if (doUpdateTFT) { UpdateTFT(); }

  // frames mitzählen 
  frameNr++;
}

// ****** PAL Decoding ******

void CaptureData(word inputNr, word ADCPin, word pPic[40][38], byte pVal[])
{
  // VSYNC suchen, max. 20 ms
  VSYNCok[inputNr] = SearchVSYNC(inputNr, ADCPin, pVal, pPic);
  if (VSYNCok[inputNr]) 
  {  
    // Field Raw Daten zunächst nur einlesen  
    ReadSync(inputNr, ADCPin, pVal, pPic, HSYNC_CAPTURE);
  }
  else { missedVSYNCcount[inputNr]++; }  
  
}

void DecodeData(word inputNr, word ADCPin, word pPic[40][38], byte pVal[])
{
  if (VSYNCok[inputNr]) 
  {  
    // Field dekodieren  
    HSYNCok[inputNr] = ReadSync(inputNr, ADCPin, pVal, pPic, HSYNC_DECODE);
    if (!HSYNCok[inputNr]) { missedHSYNCcount[inputNr]++; }
  }  
}

bool SearchVSYNC(word inputNr, word ADCPin, byte pVal[], word pPic[40][38])
{
  bool found = false;
  t1 = micros();  

  // max. 20 ms auf VSYNC warten
  while ((!found) && (micros()-t1 < 20000))
  {
    found = ReadSync(inputNr, ADCPin, pVal, pPic, VSYNC);
  }  
  // Wartezeit loggen
  VSYNCwait_us[inputNr] = micros()-t1;
  
  return found;
}

// Zeitbedarf ReadSync ist timeout_ns + ca. 7-20us
bool ReadSync(word inputNr, word ADCPin, byte pVal[], word pPic[40][38], word syncType) 
{
  unsigned long analyzeSampleCount = 0;
  word lowCnt  = 0;
  word highCnt = 0;
  unsigned long lowTime_ns  = 0;
  unsigned long highTime_ns = 0;
  bool syncFrqHigh         = false;
  bool lowDurationShort    = false;
  unsigned long timeout_ns  = 0;
  bool durationOK           = false;
  bool distanceOK           = false;
  word syncCount            = 0;
  word validOscCnt          = 0;
  bool doReadData           = false;
  bool doDecodeData         = false;
  bool result               = false;
  const word lowDurationThr_ns   =  8150;
  const word syncDistanceThr_ns  = 48000;
  word syncThr              = 0;
  int i;
  bool isEvenFieldLine      = true;
  word picLineNr            = 0;
  word quaterCnt            = 0;
  
  // PAL timings
  switch (syncType) 
  {
    case VSYNC:         lowDurationShort = false; syncFrqHigh = true;  syncCount = 2;           timeout_ns = (syncCount+1) * 32000; doReadData = true; doDecodeData = true; break;  // 30us / 2us        
    case HSYNC_CAPTURE: lowDurationShort = true;  syncFrqHigh = false; syncCount = 303;         timeout_ns = 19760000; doReadData = true;  doDecodeData = false; break;             // 4.7 ys / 59.3 us
    case HSYNC_DECODE:  lowDurationShort = true;  syncFrqHigh = false; syncCount = 150; /*303*/ timeout_ns = 19760000; doReadData = false; doDecodeData = true; break;             
  }

  // zu analysierende SampleMenge berechnen
  analyzeSampleCount = timeout_ns / sampleTime_ns +1;
  if (analyzeSampleCount > 14200) { analyzeSampleCount = 14200; }

  if (doReadData)
  {
    // Kanal umschalten & Dummy lesen
    changeChannel(ADCPin);
    analogReadEx();

    for(i = 0; i < analyzeSampleCount; i++)
    {
      pVal[i] = analogReadEx(); // so schnell wie möglich
    }
    
    result = true;
  }

  if (doDecodeData)
  { // D

    i = 0;
    validOscCnt = 0;
    syncThr     = syncThreshold[inputNr];
      
    // einmal initial auf Low warten
    while ((pVal[i] > syncThr) && (i < analyzeSampleCount)) { i++; }        
 
    while ((validOscCnt < syncCount) && (i < analyzeSampleCount))
    {
      lowCnt  = 0;
      highCnt = 0;

      // beim HSYNC-Decoding ungerade Zeilen des Fields überspringen, halbe vertikale Resolution reicht völlig aus
      if ((syncType == HSYNC_DECODE) && (!isEvenFieldLine))
      {
        // eine komplette Zeile überspringen
        i+= samplesPerLine;
        // wieder auf Low snychronisieren
        while ((pVal[i] > syncThr) && (i < analyzeSampleCount)) { i++; } 
      }
    
      // Low zählen  
      while ((pVal[i] <= syncThr) && (i < analyzeSampleCount)) { lowCnt++; i++; }

      // High zählen
      while ((pVal[i] > syncThr) && (i < analyzeSampleCount)) { highCnt++; i++; }
  
      // Zeiten berechnen
      lowTime_ns  = lowCnt  * sampleTime_ns; 
      highTime_ns = highCnt * sampleTime_ns;
          
      if (lowDurationShort) { durationOK = lowTime_ns <= lowDurationThr_ns; } 
                       else { durationOK = lowTime_ns >  lowDurationThr_ns; }   

      if (durationOK)
      {
        if (syncFrqHigh) { distanceOK = lowTime_ns + highTime_ns <= syncDistanceThr_ns; }   
                    else { distanceOK = lowTime_ns + highTime_ns >  syncDistanceThr_ns; } 
  
        // gesuchte Oscillation gefunden ? 
        if (distanceOK) 
        { 
          // ggf. Line lesen
          if (syncType == HSYNC_DECODE)
          {
            // picLine berechnen, ab Zeile 18
            if (validOscCnt >= 18) {
                                      if (quaterCnt == 3) { quaterCnt = 0; picLineNr++; }  // alle 4 Zeilen auf nächste picLine springen
                                                     else { quaterCnt++;                }
                                   }
             
            ReadLine(inputNr, pVal, pPic, validOscCnt, i - highCnt, highCnt, picLineNr);  
            
            // Flag für gerade/ungerade Line toggeln
            isEvenFieldLine = !isEvenFieldLine;            
          }  
          // merken, wieviele gültige Oscillation gefunden wurden
          validOscCnt++;
        }
      }
    }  
    result = validOscCnt == syncCount;

  } // D
  
  return result;
}

void ReadLine(word inputNr, byte pVal[], word pPic[40][38], word lineNumber, word pos, word lineSampleCount, word picLineNumber)
{
  byte asc = pixPerLine;
  if (asc > lineSampleCount) { asc = lineSampleCount; }  // bounding
  
  // Lese-Anfang entpricht ArrayPos + BackPorch
  unsigned long n = 0;
  unsigned long x = pos + backPorchSampleCount; 
 
  if (lineNumber < 18)    
  {
    // bis 18 Datenlose bzw. partiell datenlose Lines (ggf. Teletext)
    if (lineNumber == 0)
    {          
       realBlackLevel[inputNr] = 0;
       
       // anhand erster Zeile wird realBlackLevel ermittelt      
       while (n < asc)
       {
         realBlackLevel[inputNr] += pVal[x+n++];
       }  
       // Average aller High-Werte der 1. Zeile
       realBlackLevel[inputNr] /= n;  
    }  
  }  
  else
  {
    byte* pValue = &pVal[x];
    
    while (n < asc)
    {
       // echte Pixel übernehmen, 4 Zeilen werden zu einer zusammengefasst, Helligkeiten liegen also in pPic mit 4-fachem Gain vor
       pPic[n++][picLineNumber] += *(pValue++);   
    }
  }  
 
}  

void ProfileSampleTime(word inputNr, word ADCPin, byte pVal[])
{
  int i;
  changeChannel(ADCPin);
  analogReadEx();
    
  // SampleTime  
  t1 = micros(); 
  
  // 4000 Werte
  for(i = 0; i <= 3999; i++)
  {
    pVal[i] = analogReadEx(); // so schnell wie möglich
  }

  sampleTime_ns = (micros() - t1) / 4; 
  if (sampleTime_ns == 0) { sampleTime_ns = 1; }

  // SampleTime-anhängige Werte
  backPorchSampleCount =  5700 / sampleTime_ns +1; // backPorch berechnen (vorn), +1 um garantiert im Bild anzufangen
  samplesPerLine = 64000 / sampleTime_ns;
  pixPerLine = 51950 / sampleTime_ns;  
  if (pixPerLine > 40) { pixPerLine = 40; }
}

void ProfileSyncLevel(word inputNr, word ADCPin, byte pVal[])
{
  int i;
  changeChannel(ADCPin);
  analogReadEx();
  
  // ca. 200 us lesen, sollte mind. 3 Syncs beinhalten 
  for(i = 0; i <= 133; i++)
  {
    pVal[i] = analogReadEx(); // so schnell wie möglich
  }
      
  // Sync-Level
  word minValue = 4096;
  word tmp      = 0;
  
  for(i = 0; i < 133; i++)
  {
    // Sync-Level bestimmen
    if (pVal[i] < minValue) { minValue = pVal[i]; }
  }  
  
  // SyncLevel aus Median der letzen n Werte ermitteln, Kanal-getrennt
  switch (inputNr)
  {
    case 0: syncLevelMedianR.add(minValue);
            if (syncLevelMedianR.getMedian(tmp) == 0) { syncLevel[inputNr] = tmp; } // Sync-Threshold R berechnen
    break;
    case 1: syncLevelMedianG.add(minValue);
            if (syncLevelMedianG.getMedian(tmp) == 0) { syncLevel[inputNr] = tmp; } // Sync-Threshold G berechnen
    break;    
    case 2: syncLevelMedianB.add(minValue);
            if (syncLevelMedianB.getMedian(tmp) == 0) { syncLevel[inputNr] = tmp; } // Sync-Threshold B berechnen
    break;
  }
  
  blackLevel[inputNr]    = syncLevel[inputNr]  + VoltsToADCUnits(0.106);   // theoret. Black-Level liegt 0.3V über Sync-Level, wegen künstl. Sync-Level-Addition nur ca. 230 mv
  syncThreshold[inputNr] = (syncLevel[inputNr] + blackLevel[inputNr]) / 2;  
    
}

unsigned long VoltsToADCUnits(float voltage)
{
  return voltage * voltageGain * 255 / 3.3; // 4095
}

// ****** Image Calculations ******

void ResetPic(byte resetMode)
{
  byte redLine = 0;
  byte blueLine = 0;
  byte x, y;
  
  for(y = 0; y < 38; y++)
   for(x = 0; x < 40; x++)
   { 
     picR[x][y] = 0;
     picG[x][y] = 0;
     picB[x][y] = 0;
   }    
 
  switch (resetMode)
  {  
    case RM_BLANK: 
      // nothing to do
    break;
    case RM_TEST_PATTERN: 
              redLine = frameNr%38;
              blueLine = (frameNr+8)%38;
              for(y = 0; y < 38; y++)
               for(x = 0; x < 40; x++) 
               { 
                 if ((y>=0) && (y<blueLine))  { picR[x][y] = 0; picG[x][y] = 255*4; }
               }  
    break;               
  }             
}

void PrepareLookUps()
{
  // Interpolation vorberechnen
  float realPos = 0;
  float srcPos  = 0;
  int n, a;
  word w;

  // Gamma LookUp vorberechnen
  for(n = 0; n < 256; n++)
  {  
    gammaTableTFT[n] = 255.0 * pow(n / 255.0, 1.0 / gamma_value_TFT) + 0.5;   // anhebendes Gamma für TFT 
    gammaTableLED[n] = 255.0 * pow(n / 255.0, 1.0 / params[3][0])    + 0.5;   // abgesenktes Gamma für LEDs  , gamma_value_LED
  }  

  // TopLine ?
  if (is169)  { topLine = params[14][0]; }  // 16:9
         else { topLine = 1;             }  // 4:3
           
  // SideChains
  for(n = 0; n < params[13][0]; n++)  // sideChainCnt
  {
    // Linear-Interpolation
    realPos = (float)n / (params[13][0]-1);
    if (is169) { srcPos = params[15][0] * realPos + topLine; } // 16:9  params[15] = approx. 56
          else { srcPos = 35.0          * realPos + topLine; } // 4:3 komplettes Bild, // nicht ganz volle Höhe 
    sideChainInterpolationPos[n]   = (byte)srcPos;
    sideChainInterpolationDelta[n] = srcPos - (float)sideChainInterpolationPos[n];
    // Reset 
    for(a = 0; a < 10; a++) 
    {
     leftChain[a][n][0] = 0;
     leftChain[a][n][1] = 0;
     leftChain[a][n][2] = 0;
     rightChain[a][n][0] = 0;
     rightChain[a][n][1] = 0;
     rightChain[a][n][2] = 0;
    } 
  }
  
  // TopChain
  for(n = 0; n < params[12][0]; n++)  // topChainCnt
  {
    // Linear-Interpolation
    realPos = (float)n / (params[12][0]-1);
    srcPos  = (pixPerLine-1) * realPos;
    topChainInterpolationPos[n]   = (byte)srcPos;
    topChainInterpolationDelta[n] = srcPos - (float)topChainInterpolationPos[n];
    // Reset
    for(a = 0; a < 10; a++) 
    {    
     topChain[a][n][0] = 0;
     topChain[a][n][1] = 0;
     topChain[a][n][2] = 0;    
    }
  }
   
}
  
void TransformPic(bool fully)
{
  short r,g,b;
  float coeff;
  word rblR4 = realBlackLevel[0] * 4;
  word rblG4 = realBlackLevel[1] * 4;
  word rblB4 = realBlackLevel[2] * 4;
  byte x, y; 
  short firstContentPixelTop, firstContentPixelBottom;
  const byte thr169 = 12;              // über 12% Brightness wird als Content gewertet
  const byte ratioChangeFrameCnt = 60; // nach 60 Frames ggf. Ratio-Mode-Wechsel
  bool old169;
  const byte pplm = pixPerLine-1;

    if ((VSYNCok[0]) || (VSYNCok[1]) || (VSYNCok[2]))
    {
      // mind. ein VSYNC erkannt, Bilddaten extrahieren
      
      coeff = params[2][0];  // input_contrast_percent
     
      // Center-Color
      byte pos = pixPerLine / 2 - 8; // 4 mit 2er Abstand auf jeder Seite links/rechts
      centerColor[avrgIndex][0] = 0;
      centerColor[avrgIndex][1] = 0;
      centerColor[avrgIndex][2] = 0;
      for(x = 0; x < 8; x++) 
      {
        // Lines 13 & 21 mitteln
        r = picR[pos][13] - rblR4; if (r < 0) { r = 0; } centerColor[avrgIndex][0] += r;  
        g = picG[pos][13] - rblG4; if (g < 0) { g = 0; } centerColor[avrgIndex][1] += g;
        b = picB[pos][13] - rblB4; if (b < 0) { b = 0; } centerColor[avrgIndex][2] += b;
        r = picR[pos][21] - rblR4; if (r < 0) { r = 0; } centerColor[avrgIndex][0] += r; 
        g = picG[pos][21] - rblG4; if (g < 0) { g = 0; } centerColor[avrgIndex][1] += g;
        b = picB[pos][21] - rblB4; if (b < 0) { b = 0; } centerColor[avrgIndex][2] += b;           
        pos += 2;
      }   
      // auf RGB 0-255 bringen
      centerColor[avrgIndex][0] = (centerColor[avrgIndex][0] >> 4) * coeff;  if (centerColor[avrgIndex][0] > 255) { centerColor[avrgIndex][0] = 255; }
      centerColor[avrgIndex][1] = (centerColor[avrgIndex][1] >> 4) * coeff;  if (centerColor[avrgIndex][1] > 255) { centerColor[avrgIndex][1] = 255; }
      centerColor[avrgIndex][2] = (centerColor[avrgIndex][2] >> 4) * coeff;  if (centerColor[avrgIndex][2] > 255) { centerColor[avrgIndex][2] = 255; }

      // Lines oder komplett-Bild    
      for(y = 0; y < 38; y++)
       for(x = 0; x < pixPerLine; x++) 
       {
         if ((fully) || ((!fully) && ((x == 0) || (x == pplm) || (y == topLine))  ))
         {    
          // 1/4 Luma, wegen Zeilenaddition, blackLevel abziehen, Scaling (contrast)
          r = (picR[x][y] - rblR4) * coeff + 0.5;
          g = (picG[x][y] - rblG4) * coeff + 0.5;
          b = (picB[x][y] - rblB4) * coeff + 0.5;       
       
          // bounding
          if (r < 0) { r = 0; } else if (r > 255) { r = 255; }
          if (g < 0) { g = 0; } else if (g > 255) { g = 255; }
          if (b < 0) { b = 0; } else if (b > 255) { b = 255; }
       
          // übernehmen
          picR[x][y] = r; 
          picG[x][y] = g;
          picB[x][y] = b;
         }
       } 

    }
    else
    {
               
      for(y = 0; y < 38; y++)
       for(x = 0; x < pixPerLine; x++) 
       {
         picR[x][y] = 64; 
         picG[x][y] = 64;
         picB[x][y] = 128;       
       }
    }

   // 16:9 Erkennung
   firstContentPixelTop    = -1;
   firstContentPixelBottom = 99;
   y = 0;
   while ((y < 8) && (firstContentPixelTop == -1))    
   {
     if ((picR[19][y] > thr169) || (picG[19][y] > thr169) || (picB[19][y] > thr169)) { firstContentPixelTop = y; }
     y++;
   }
   y = 37;
   while ((y > 28) && (firstContentPixelBottom == 99))
   {
     if ((picR[17][y] > thr169) || (picG[17][y] > thr169) || (picB[17][y] > thr169)) { firstContentPixelBottom = y; }
     y--;
   } 

   if ((firstContentPixelTop != -1) && (firstContentPixelBottom != 99))
   {
     // beide Werte müssen gültig sein, damit Ratio-Entscheidung mgl. ist 
     if ((firstContentPixelTop >= 3) && (firstContentPixelBottom <= 35)) { frameCnt169++;  if (frameCnt169 > ratioChangeFrameCnt) { frameCnt169 = ratioChangeFrameCnt; }  }  // ist 16:9 frame
                                                                    else { frameCnt169--;  if (frameCnt169 < 0)                   { frameCnt169 = 0;                   }  }  // ist 4:3 frame
   }                                                                
   
   // ist 16:9 Modus, wenn einige Sekunden schwarze Balken erkannt wurden                                                              
   old169 = is169;
   if (frameCnt169 == ratioChangeFrameCnt) { is169 = true;  } else
   if (frameCnt169 == 0)                   { is169 = false; }
   // bei Ratio-Wechsel LookUps neu berechnen
   if (is169 != old169) { PrepareLookUps(); }
                                                              
}  

void UpdateLEDs()
{
  float d,d2;
  byte p1,p2,rt,top;
  word r1,g1,b1,r2,g2,b2;
  int n, a;
  byte aCnt         = params[10][0];
  byte topChainCnt  = params[12][0];
  byte sideChainCnt = params[13][0];
  byte sn1;
  byte tn1;
 
  // SideChain Interpolation inkl. zeitl. Epsilon
  if (pixPerLine > 0) { rt = pixPerLine -1; }
                 else { rt = 0; }
  for(n = 0; n < sideChainCnt; n++)
  {
    sn1 = sideChainCnt-n-1;
    d  = sideChainInterpolationDelta[n];
    d2 = 1.0-d;
    p1 = sideChainInterpolationPos[n];
    p2 = p1 +1;
    leftChain[avrgIndex][n][0] = (d2 * picR[0][p1] + d * picR[0][p2]) + 0.5;  // R
    leftChain[avrgIndex][n][1] = (d2 * picG[0][p1] + d * picG[0][p2]) + 0.5;  // G
    leftChain[avrgIndex][n][2] = (d2 * picB[0][p1] + d * picB[0][p2]) + 0.5;  // B
    rightChain[avrgIndex][sn1][0] = (d2 * picR[rt][p1] + d * picR[rt][p2]) + 0.5;  // R
    rightChain[avrgIndex][sn1][1] = (d2 * picG[rt][p1] + d * picG[rt][p2]) + 0.5;  // G
    rightChain[avrgIndex][sn1][2] = (d2 * picB[rt][p1] + d * picB[rt][p2]) + 0.5;  // B    

    r1 = 0;  g1 = 0;  b1 = 0;  r2 = 0;  g2 = 0;  b2 = 0;
    // Averaging, Farbanteile einzeln
    for(a = 0; a < aCnt; a++) 
    {
      r1 += leftChain[a][n][0];
      g1 += leftChain[a][n][1];
      b1 += leftChain[a][n][2];
      r2 += rightChain[a][n][0];
      g2 += rightChain[a][n][1];
      b2 += rightChain[a][n][2];
    }
    leftChainOut[n][0] =  r1 / aCnt + 0.5;
    leftChainOut[n][1] =  g1 / aCnt + 0.5;
    leftChainOut[n][2] =  b1 / aCnt + 0.5;
    rightChainOut[n][0] = r2 / aCnt + 0.5;
    rightChainOut[n][1] = g2 / aCnt + 0.5;
    rightChainOut[n][2] = b2 / aCnt + 0.5;    
  }
 
  top = topLine;
  for(n = 0; n < topChainCnt; n++)
  {
    tn1 = topChainCnt-n-1;
    d  = topChainInterpolationDelta[n];
    d2 = 1.0-d;    
    p1 = topChainInterpolationPos[n];
    p2 = p1 +1;
    topChain[avrgIndex][tn1][0] = (d2 * picR[p1][top] + d * picR[p2][top]) + 0.5;  // R
    topChain[avrgIndex][tn1][1] = (d2 * picG[p1][top] + d * picG[p2][top]) + 0.5;  // G
    topChain[avrgIndex][tn1][2] = (d2 * picB[p1][top] + d * picB[p2][top]) + 0.5;  // B

    r1 = 0;  g1 = 0;  b1 = 0; 
    // Averaging, Farbanteile einzeln
    for(a = 0; a < aCnt; a++) 
    {
      r1 += topChain[a][n][0];
      g1 += topChain[a][n][1];
      b1 += topChain[a][n][2];
    }
 
    topChainOut[n][0] =  r1 / aCnt + 0.5;
    topChainOut[n][1] =  g1 / aCnt + 0.5;
    topChainOut[n][2] =  b1 / aCnt + 0.5;    
  }

  // Center Color 
  r1 = 0;  g1 = 0;  b1 = 0; 
  // Averaging, Farbanteile einzeln
  for(a = 0; a < aCnt; a++) 
  {
    r1 += centerColor[a][0];
    g1 += centerColor[a][1];
    b1 += centerColor[a][2];
  }  
  centerColorOut[0] =  r1 / aCnt + 0.5;
  centerColorOut[1] =  g1 / aCnt + 0.5;
  centerColorOut[2] =  b1 / aCnt + 0.5; 
  // Center Brightness
  if (centerColorOut[0]  > centerColorOut[1]) { centerV = centerColorOut[0]; } else { centerV = centerColorOut[1]; }
  if (centerV            > centerColorOut[2]) {                            ; } else { centerV = centerColorOut[2]; }
               
  // im Avrg-Feld eins weiter gehen (0 bis max 9)
  avrgIndex++;
  avrgIndex %= aCnt;

  ShowLEDs(); 
}

void ShowLEDs()
{
  int n;
  byte topChainCnt     = params[12][0];
  byte sideChainCnt    = params[13][0];
  byte centerColAmount = params[11][0] * 255;
 
  // Center Color
  ProcessLEDfast(centerColorOut, params[4][0], params[5][0], params[7][0], params[8][0], params[9][0], centerColorOut, 0, 0); 

  // Right & Left
  for(n = 0; n < sideChainCnt; n++)
  { 
    ProcessLEDfast(rightChainOut[n], params[4][0], params[5][0], params[7][0], params[8][0], params[9][0], centerColorOut, centerV, centerColAmount);   //Debug tft.println("RGB " + String(rightChain[n][0]) + " " + String(rightChain[n][1]) + " " + String(rightChain[n][2]));
    strip.setPixelColor(n, rightChainOut[n][0], rightChainOut[n][1], rightChainOut[n][2]);

    ProcessLEDfast(leftChainOut[n], params[4][0], params[5][0], params[7][0], params[8][0], params[9][0], centerColorOut, centerV, centerColAmount);
    strip.setPixelColor(sideChainCnt + topChainCnt + n, leftChainOut[n][0], leftChainOut[n][1], leftChainOut[n][2]);  
  }  
  // Top 
  for(n = 0; n < topChainCnt; n++)
  {

    ProcessLEDfast(topChainOut[n], params[4][0], params[5][0], params[7][0], params[8][0], params[9][0], centerColorOut, centerV, centerColAmount);
    strip.setPixelColor(sideChainCnt + n, topChainOut[n][0], topChainOut[n][1], topChainOut[n][2]);  
  }   
   
  strip.show(); // LED update
}

// ****** GUI ******

void ReadKnobs()
{
  word btnV, potV;
  word tmp = 0;
  bool oldBtnPressed;
  word oldPotValue;
  float pdelta, newV;
   
  // Button
  changeChannel(analogPinButton);
  analogReadEx();  
  btnV = analogReadEx();  
   
  // Auswertung
  oldBtnPressed = buttonPressed;
  buttonPressed = btnV < 128;  // 8 bit
  if ((buttonPressed) && (!oldBtnPressed))
  {
    // Press-Event toggled durch 15 Params, 0 = OFF
    selectedParam++;
    selectedParamPotiOrig = 0; // Poti-Zustand als unverändert markieren
    if (selectedParam > 15) {
                              // bei Wechsel in Settings-OFF Zustand alle Params speichern       
                              selectedParam = 0;
                              SaveParamsToSDCard(); 
                            }
    triggerRepaint = true;
    tLastRepaintEvent = millis();
  }

  // Poti nur im Settings-Mode auslesen
  if (selectedParam > 0) 
  {

    // Poti
    changeChannel(analogPinPoti); 
    analogReadEx();
    // ADC Zeit geben zum Einpegeln geben, 100kOhm Poti
    potV = 0;
    for(int n = 0; n < 4000; n++)
    {
      potV += analogReadEx(); 
    }  
    potV = potV / 4000 + 0.5;
    potiMedian.add(potV);  // word(potV/4)*4+0.5, auf 64 diskrete Stufen bringen
      
    oldPotValue = potiValue;
    if (potiMedian.getMedian(tmp) == 0) { 
                                          potiValue = tmp;
                                          if (selectedParamPotiOrig == 0) { selectedParamPotiOrig = potiValue; }
                                          if (potiValue != oldPotValue)
                                          { 
                                            // Wert in Param übernehmen
                                            pdelta = params[selectedParam][2] - params[selectedParam][1];
                                            newV = pdelta * potiValue / 255.0 + params[selectedParam][1];
                                            if ((selectedParam < 2) || (selectedParam == 10) || (selectedParam > 11)) { newV = (unsigned long)(newV+0.5); } // ggf. Param quantisieren

                                            if (selectedParamPotiOrig > 990) {  
                                                                                params[selectedParam][0] = newV;
                                                                                // Repaint Event auslösen
                                                                                triggerRepaint = true;
                                                                                tLastRepaintEvent = millis();
                                                                             }
                                                                        else { 
                                                                               // initial muß Poti 5% bewegt werden, um Änderung zu triggern  (5% von 255 = 12)                                                                       
                                                                               if (abs(selectedParamPotiOrig - potiValue) > 12) 
                                                                               {          
                                                                                 // poti hat sich gedreht -> ab jetzt ist Wert änderbar 
                                                                                 selectedParamPotiOrig = 999;
                                                                               }
                                                                             }
                                          
                                            // topChainCnt, sideChainCnt, 16:9 topLine, 16:9 height sofort anwenden
                                            if (selectedParam > 9) { PrepareLookUps(); }  
                                          }
                                        }       
    // ggf. Settings wieder zuklappen
    if (millis() - tLastRepaintEvent > 15000)
    {
      selectedParam  = 0;
      SaveParamsToSDCard();
      triggerRepaint = true;
      tLastRepaintEvent = millis();
    }                                                         
  }                                      


}

void UpdateTFT()
{

  const word ST7735_GRAY     = 0xCCCC;
  const word ST7735_DARKGRAY = 0x7BEF;
  const word dispYScale  = 2; //2;
  unsigned long td;    
  int showInputNr = 0;
  word waveColor  = 0;
  byte r,g,b, x, y;
  byte showStart, showCnt;
  int i;
  word wl, fps;
  
  if (selectedParam > 0)
  {
      tft.fillRect(0, 0, 90, 127, ConvertRGB24to565(0, 150, 255));
      tft.fillRect(90, 0, 159, 127, ConvertRGB24to565(0, 170, 255));

      if (selectedParam < 12) { showStart = 0;  showCnt = 11; }  // Page 1
                         else { showStart = 11; showCnt = 4;  }  // Page 2
      
      // Settings anzeigen
      for(y = showStart; y < showStart + showCnt; y++)
      {
         
         if (y+1 == (selectedParam)) { 
                                       tft.fillRect(0, (y-showStart)*10, 159, 10, ST7735_WHITE);
                                       tft.setTextColor(ConvertRGB24to565(0, 150, 255));
                                     }
                                else { 
                                       tft.setTextColor(ST7735_WHITE);
                                     }
                                      
         tft.setCursor(2, (y-showStart)*10+2);                       
         tft.println(paramNames[y+1]);
         tft.setCursor(92, (y-showStart)*10+2);                       
         tft.println(String(params[y+1][0]));
      }

  }
  else
  {
    
    if (params[1][0] > 0.5)
    {
      
      tft.fillScreen(ST7735_BLACK);
      
      // Debug Screen anzeigen
      if ((VSYNCok[2]) && (HSYNCok[2])) { showInputNr = 2; waveColor = ST7735_BLUE;  } else
      if ((VSYNCok[1]) && (HSYNCok[1])) { showInputNr = 1; waveColor = ST7735_GREEN; } else
      if ((VSYNCok[0]) && (HSYNCok[0])) { showInputNr = 0; waveColor = ST7735_RED;   } else
                                        { showInputNr = 1; waveColor = ST7735_GRAY;  }
      
      tft.setTextColor(ST7735_WHITE);
    
      // WhiteLevel 0.7 Volt über BlackLevel
      wl = realBlackLevel[showInputNr] + VoltsToADCUnits(0.7);      
      tft.drawLine(0, 127-wl/dispYScale, 159, 127-wl/dispYScale, ST7735_DARKGRAY);
      
      // realBlackLevel
      tft.drawLine(0, 127-realBlackLevel[showInputNr]/dispYScale, 159, 127-realBlackLevel[showInputNr]/dispYScale, ST7735_DARKGRAY);
      
      // BlackLevel
      tft.drawLine(0, 127-blackLevel[showInputNr]/dispYScale, 159, 127-blackLevel[showInputNr]/dispYScale, ST7735_GRAY);
      
      // Sync-Threshold
      tft.drawLine(0, 127-syncThreshold[showInputNr]/dispYScale, 159, 127-syncThreshold[showInputNr]/dispYScale, ST7735_WHITE);
    
      // Analog-Signal
      switch(showInputNr)
      {
        case 0: for(i = 0; i <= 158; i++) { tft.drawLine(i, 127-valR[i+810]/dispYScale, i+1, 127-valR[i+810+1]/dispYScale, waveColor); } break; // ca. 19 Zeilen 1216us into the Signal = 810 
        case 1: for(i = 0; i <= 158; i++) { tft.drawLine(i, 127-valG[i+810]/dispYScale, i+1, 127-valG[i+810+1]/dispYScale, waveColor); } break;
        case 2: for(i = 0; i <= 158; i++) { tft.drawLine(i, 127-valB[i+810]/dispYScale, i+1, 127-valB[i+810+1]/dispYScale, waveColor); } break;
      }
     
      // Input-Pixels
      for(y = 0; y < 38*2; y++)
       for(x = 0; x < pixPerLine*2; x++) 
       { 
         r = picR[x/2][y/2];  r = gammaTableTFT[r];
         g = picG[x/2][y/2];  g = gammaTableTFT[g];
         b = picB[x/2][y/2];  b = gammaTableTFT[b];  
         tft.drawPixel(x, y+52, ConvertRGB24to565(r, g, b)); 
       }  
       
       // LED-Pixels (TopLine)
       for(x = 0; x < params[12][0]; x++)
       {
         tft.fillRect(120+params[12][0]-x*2, 110, 2, 4, ConvertRGB24to565(topChainOut[x][0], topChainOut[x][1], topChainOut[x][2]));  
       }   
       
       // Center-Color
       tft.fillRect(150, 120, 8, 8, ConvertRGB24to565(centerColorOut[0], centerColorOut[1], centerColorOut[2]));  
        
      tft.setCursor(0, 0);
      tft.println("SampleTime: " + String(sampleTime_ns) + " ns");
      tft.println("SyncLevel:  " + String(syncLevel[0]) + " " + String(syncLevel[1]) + " " + String(syncLevel[2]));
      //tft.println("BlackLevel:  " + String(blackLevel));  // berechnet
      //tft.println("BlackLevel: "  + String(realBlackLevel[0]) + " " + String(realBlackLevel[1]) + " " + String(realBlackLevel[2]));  // real
      //tft.println("WhiteLevel:  " + String(whiteLevel));   
      tft.println("bad VSYNCs: " + String(missedVSYNCcount[0])+ " " + String(missedVSYNCcount[1]) + " " + String(missedVSYNCcount[2])); 
      tft.println("bad HSYNCs: " + String(missedHSYNCcount[0])+ " " + String(missedHSYNCcount[1]) + " " + String(missedHSYNCcount[2])); 
      tft.println("VSYNC wait ms:  " + String(VSYNCwait_us[0]/1000)+ " " + String(VSYNCwait_us[1]/1000) + " " + String(VSYNCwait_us[2]/1000)); 
      //tft.println("validLines:  " + String(validOscCnt)); 
      td = t4-t3;
      if (td > 0) {  tft.println("Timing: " + String((t4-t3)/1000.0) + " ms " + String((word)(1000000.0/(t4-t3) /*+0.5*/)) + " fps"); }
      tft.setCursor(73, 53);
      if (is169) { tft.println("16:9"); } else { tft.println("4:3"); }
    }  
    else
    {
       if (triggerRepaint)
       {      
        tft.fillScreen(ST7735_BLACK);
        tft.setCursor(36, 58);
        tft.setTextColor(ST7735_WHITE);
        tft.println("Ambitious Light");
        tft.setTextColor(ST7735_DARKGRAY);
        lastFPS = 0;
       }

       // im Normal-Betrieb nur fps anzeigen
       fps = (word)(1000000/(t4-t3)); //+0.5;
       if (fps != lastFPS)
       {
         tft.fillRect(36, 68, 40, 8, ST7735_BLACK);
         tft.setCursor(36, 68);
         tft.println(String(fps) + " fps");
         lastFPS = fps;
       }  
    }
  }

  // Repaint-Event zurücksetzen
  triggerRepaint = false;
}


// ******************* SDCard *****************

void LoadParamsFromSDCard()
{
    // re-open the file for reading:
    paramFile = SD.open(paramfileName);
    if (paramFile)
    {
       byte fourByte[4];
       for(int n = 0; n < 16; n++)
       {
         if (paramFile.available() >= 4)
         {
           // 4 Bytes lesen und daraus einen Float machen
           fourByte[0] = paramFile.read();
           fourByte[1] = paramFile.read();
           fourByte[2] = paramFile.read();
           fourByte[3] = paramFile.read();
           params[n][0] = *((float*)(&fourByte[0]));
           // MinMax sicherstellen
           if (params[n][0] < params[n][1]) { params[n][0] = params[n][1]; }
           if (params[n][0] > params[n][2]) { params[n][0] = params[n][2]; }
         }
       }    
       paramFile.close();
    }
    else
    { 
      tft.setTextColor(ST7735_RED);
      tft.println("Load params failed");
      delay(3000);
    } 
}

void SaveParamsToSDCard()
{
    tft.fillScreen(ST7735_BLACK);
    tft.setCursor(0, 0);
    tft.setTextColor(ST7735_WHITE);
    tft.println("Saving params ...");
         
    // open for writing
    paramFile = SD.open(paramfileName, FILE_WRITE);
    if (paramFile)
    {
     
       paramFile.seek(0);    
       byte fourByte[4];
       for(int n = 0; n < 16; n++)
       {
          // float to 4 Bytes 
          unsigned long l = *(unsigned long*)(&params[n][0]);
          fourByte[0] = l & 0x00FF;
          fourByte[1] = (l >> 8) & 0x00FF;
          fourByte[2] = (l >> 16) & 0x00FF;
          fourByte[3] = l >> 24;
          // 4 Bytes schreiben     
          paramFile.write(fourByte, 4);
       }
       paramFile.close();
    }
    else
    { 
      tft.setTextColor(ST7735_RED);
      tft.println("Save params failed");
      delay(3000);
    }     
}

// ******************* Color Space Conversion *********************

word ConvertRGB24to565(byte red, byte green, byte blue)
{
  return ((red / 8) << 11) | ((green / 4) << 5) | (blue / 8);
}

void ProcessLEDfast(byte rgb[], float satRel, float brightRel, float redRel, float greenRel, float blueRel, byte centerCol[], byte centerVal, byte centerAmount)
{
    word minimum, maximum, delta, v, ff; 
    unsigned long r = rgb[0];
    unsigned long g = rgb[1];
    unsigned long b = rgb[2];
    unsigned long s, p, q, t;
    int h;

    // min
    if (r  < g) { minimum = r; } else { minimum = g; }
    if (minimum < b) {       ; } else { minimum = b; }

    // max
    if (r  > g) { maximum = r; } else { maximum = g; }
    if (maximum > b) {       ; } else { maximum = b; }

    if(maximum > 0)
    {                              
      delta = (maximum - minimum);      

      if (delta > 0)
      {
        s = ((delta << 12) / maximum); 
                   
        if( r >= maximum ) { h =        ((g - b) << 10) / delta; } else  // yellow  <-> magenta
        if( g >= maximum ) { h = 2048 + ((b - r) << 10) / delta; } else  // cyan    <-> yellow
                           { h = 4096 + ((r - g) << 10) / delta; }       // magenta <-> cyan
        if (h < 0) { h += 6144; } //1536.0; }
  
        // ggf. Saturation anwenden
        if (satRel < 1.0) {  s = s * satRel; }  //if (s > 4096) { s = 4096; }
      }
      else
      {
        s = 0;
        h = 0;  
      }
      
      // ggf. Brightness
      if (brightRel == 1.0) { v = maximum; }  
                       else { v = maximum * brightRel + 0.5; }
                       
      // back conversion
      if (s == 0)   { r = v; g = v; b = v; }  // grey
               else {
                      unsigned long lh = h >> 10; 
                      ff = h % 1024; 
                      p  = (v * (4190208 - (s * 1023)))        / 4190208;  // s läuft 0-4096, h und ff laufen von 0-1023, p,q,t von 0-1023
                      q  = (v * (4190208 - (s * ff)))          / 4190208; 
                      t  = (v * (4190208 - (s * (1023 - ff)))) / 4190208;
                        
                      switch(lh) {
                       case 0:    r = v; g = t; b = p;  break;
                       case 1:    r = q; g = v; b = p;  break;
                       case 2:    r = p; g = v; b = t;  break;
                       case 3:    r = p; g = q; b = v;  break;
                       case 4:    r = t; g = p; b = v;  break;
                       case 5:    r = v; g = p; b = q;  break;
                      }
                    } 
      
      // ggf. Farbanteile reduzieren
      if (redRel   < 1.0) { r = r * redRel   + 0.5; }
      if (greenRel < 1.0) { g = g * greenRel + 0.5; }                             
      if (blueRel  < 1.0) { b = b * blueRel  + 0.5; }
       
      // begrenzen
      if (r > 255) { r = 255; }
      if (g > 255) { g = 255; }
      if (b > 255) { b = 255; }
     
      // gamma
      rgb[0] = gammaTableLED[r];
      rgb[1] = gammaTableLED[g];
      rgb[2] = gammaTableLED[b];                    
    } 
    else
    {
      // schwarz
      v = 0;
    }

    if (centerAmount > 0)
    {
      // ggf. CenterColor applizieren
      word d1 = (v * 255) / centerAmount; if (d1 > 255) { d1 = 255; }
      byte d2 = 255 - v;
      rgb[0] = (d1 * rgb[0] + d2 * centerCol[0]) / 255 +0.5;
      rgb[1] = (d1 * rgb[1] + d2 * centerCol[1]) / 255 +0.5;
      rgb[2] = (d1 * rgb[2] + d2 * centerCol[2]) / 255 +0.5;
    }
    
}

